--[[
    Author: PeterAH (Modding-Welt)
    Mod name: Tire Sound
    Version: 1.0.0.0 (FS22)
    Date: January 2023
    Contact: https://www.modding-welt.com
    Discord: [MW] PeterAH#5807
]]

tireSound = {}

gU_lastDeleteTime = 0
gU_lastLoadTime = 0
gU_deleteFilename = "."
gU_deleteHandle = "."
gU_loadFilename = "."
gU_loadHandle = "."

local _modDirectory = g_currentModDirectory
local _modName = g_currentModName

function tireSound.prerequisitesPresent(specializations)
    return true
end

function tireSound.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onPreLoad", tireSound)
    SpecializationUtil.registerEventListener(vehicleType, "onPostLoad", tireSound)
    SpecializationUtil.registerEventListener(vehicleType, "onStartMotor", tireSound)
    SpecializationUtil.registerEventListener(vehicleType, "onStopMotor", tireSound)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdate", tireSound)
    SpecializationUtil.registerEventListener(vehicleType, "saveToXMLFile", tireSound)
    SpecializationUtil.registerEventListener(vehicleType, "onPreDelete", tireSound)
end

function tireSound:initSpecialization()
    local schemaSavegame = Vehicle.xmlSchemaSavegame
    schemaSavegame:register(XMLValueType.INT, "vehicles.vehicle(?)." .. _modName .. ".tireSound#configurationId", "Tire sound configuration id", 1)
end

function tireSound:onPreLoad(savegame)
    self.loadSounds = Utils.appendedFunction(Motorized.loadSounds, tireSound.loadSounds)
    
    local configurationId = Utils.getNoNil(self.configurations["tireSound"], 0)
    if savegame ~= nil then
        if configurationId > 0 then
            configurationId = savegame.xmlFile:getValue(savegame.key .. "." .. _modName .. ".tireSound#configurationId", configurationId)
            if configurationId < 1 or configurationId > 12 then
                configurationId = 1
            end
            self.configurations["tireSound"] = configurationId
        end
    end

end

function tireSound:loadSounds(xmlFile, motorId)
    -- Every vehicleReload is calling this function two times

    if self.isClient and self.spec_motorized.samples.tires == nil then
        local spec = self.spec_motorized
        local baseString = "sounds"
        local sample = spec.samples
        local xmlSoundFile

        if fileExists(_modDirectory .. "sounds/tireSound.xml") then
            xmlSoundFile = loadXMLFile("tireSound", _modDirectory .. "sounds/tireSound.xml")
        else
            print("Error: Script tireSound.lua: XML-File not found: " .. _modDirectory .. "sounds/tireSound.xml")
        end
        
        local outInMultiplier
        if xmlSoundFile ~= nil then
            -- Adjust soundRadius to the soundRadius of the mod + extra distance because this sound can be heard far away
            if type(spec.motorSamples) == "table" then
                for key,_ in ipairs(spec.motorSamples) do
                    local sample = spec.motorSamples[key]
                    if sample.isGlsFile ~= nil and sample.isGlsFile == true and sample.templateName ~= nil then
                        local temp = sample.templateName
                        if temp == "engineSmall" or temp == "engineSmallVintage" or temp == "engineMedium" or temp == "engineMediumVintage" or temp == "engineLarge" then
                            if sample.outdoorAttributes.volume > 0.01 then
                                setXMLFloat(xmlSoundFile, "sounds.tires#innerRadius", sample.innerRadius + 3)
                                setXMLFloat(xmlSoundFile, "sounds.tires#outerRadius", sample.outerRadius + 15)
                                outInMultiplier = sample.indoorAttributes.volume / sample.outdoorAttributes.volume
                                break
                            end
                        end
                    end
                end
            end
            if outInMultiplier == nil and sample.motorStart ~= nil and sample.motorStop ~= nil then
                setXMLFloat(xmlSoundFile, "sounds.tires#innerRadius", sample.motorStart.innerRadius + 3)
                setXMLFloat(xmlSoundFile, "sounds.tires#outerRadius", sample.motorStart.outerRadius + 15)
            end
            spec.samples.tires = g_soundManager:loadSampleFromXML(xmlSoundFile, baseString, "tires", _modDirectory, self.components, 0, AudioGroup.VEHICLE, self.i3dMappings, self)

            -- Valtra N fix (inits wrong volume scale, so correct it through XML)
            local volScale = Utils.getNoNil(getXMLFloat(xmlSoundFile, "sounds.tires#volumeScale"), 0)
            if volScale > 0.001 then
                spec.samples.tires.volumeScale = volScale
            end
        end

        if spec.samples.tires ~= nil then 
            if outInMultiplier == nil then
                if type(spec.samples) == "table" then
                    if spec.samples.motorStart ~= nil then
                        local sample = spec.samples.motorStart
                        if sample.outdoorAttributes.volume > 0.01 then
                            outInMultiplier = sample.indoorAttributes.volume / sample.outdoorAttributes.volume
                        end
                    end
                end
            end
            if outInMultiplier ~= nil then
                if outInMultiplier > 0.98 then
                    outInMultiplier = 1
                elseif outInMultiplier > 0.89 then  -- vintage gls is 0.9
                    outInMultiplier = outInMultiplier * 0.91
                elseif outInMultiplier > 0.59 then  -- default gls is 0.8
                    outInMultiplier = outInMultiplier * 0.8125  -- sets 0.8 to 0.68
                elseif outInMultiplier > 0.54 then
                    outInMultiplier = outInMultiplier * 0.92
                elseif outInMultiplier < 0.45 then   -- old FS19 motor is 0.25
                    outInMultiplier = 0.45
                end
                
                -- consider, if the user is adjusting indoorVolume in the tireSound.xml    -- 0.65 = neutral
                if spec.samples.tires.outdoorAttributes.volume > 0.01 then   -- avoid div/0 error
                    outInMultiplier = outInMultiplier * (spec.samples.tires.indoorAttributes.volume / spec.samples.tires.outdoorAttributes.volume * 1.5385) -- 0.65 = neutral
                end
                
                spec.samples.tires.indoorAttributes.volume = spec.samples.tires.outdoorAttributes.volume * outInMultiplier
            end
            self.origIndoorVolume = spec.samples.tires.indoorAttributes.volume
            self.origOutdoorVolume = spec.samples.tires.outdoorAttributes.volume

            -- Adjust pitch to the maximum tractor speed
            local maxSpeed = math.max(spec.motor.maxForwardSpeed * 3.6, 35)
            sample.tires.indoorAttributes.pitch = sample.tires.indoorAttributes.pitch * (1 + (maxSpeed - 53) / 180) * 1.35
            sample.tires.outdoorAttributes.pitch = sample.tires.outdoorAttributes.pitch * (1 + (maxSpeed - 53) / 180) * 1.35

            local DIFFERENTIAL_SPEED = 11               -- was: SPEED = 17 (LS22), SPEED = 10 (LS19)
            -- Overwrite pitch modifier, table[11]=DIFFERENTIAL_SPEED
            sample.tires.modifiers.pitch[DIFFERENTIAL_SPEED].keyframes[2].time = maxSpeed
            sample.tires.modifiers.pitch[DIFFERENTIAL_SPEED].maxTime = maxSpeed

            -- Overwrite volume modifier, table[11]=DIFFERENTIAL_SPEED
            sample.tires.modifiers.volume[DIFFERENTIAL_SPEED].keyframes[1].time = (maxSpeed) * 0.55
            sample.tires.modifiers.volume[DIFFERENTIAL_SPEED].keyframes[2].time = maxSpeed
            sample.tires.modifiers.volume[DIFFERENTIAL_SPEED].maxTime = maxSpeed

        end
    end
end

function tireSound:onPostLoad(savegame)
    local spec = self.spec_wheels
    local numMudTires = 0
    local numNokianTires = 0
    local numCrawler = 0
    local numTiresWithSound = 0
    local isNokianTires = false
    local isCrawler = false
    local isCareWheel = false
    local wheelsMaxWidth = 0
    
    -- How many tires with sound does the vehicle have?
    for _,wheel in pairs(spec.wheels) do
        -- print(wheel.tireTrackAtlasIndex)
        if wheel.tireTrackAtlasIndex == 8 or wheel.tireTrackAtlasIndex == 10 or wheel.tireTrackAtlasIndex == 11 then
            numNokianTires = numNokianTires + 1    -- Nokian, Conti HCS, etc.
        elseif wheel.tireType == 4 then
            numCrawler = numCrawler + 1
        elseif wheel.tireTrackAtlasIndex == 0 or wheel.tireType == 1 then
            numMudTires = numMudTires + 1
        end
        wheelsMaxWidth = math.max(wheel.width, wheelsMaxWidth)
    end
    
    if numMudTires > 1 or numNokianTires > 1 or numCrawler > 1 then
        numTiresWithSound = numMudTires + numNokianTires + numCrawler
        if numCrawler >= numMudTires and numCrawler > 1 then
            isCrawler = true
        elseif numNokianTires >= numMudTires then
            isNokianTires = true
        end
    end
    
    if isCrawler == false and isNokianTires == false then
        if wheelsMaxWidth > 0 and wheelsMaxWidth < 0.41 then
            isCareWheel = true
        end
    end
    
    gU_targetSelf = nil
    if self.origIndoorVolume == nil or self.origOutdoorVolume == nil then
        if self.isClient then
            print("Error: Script tireSound: failed, sample.tires (<tires> in the tire-sound-XML) are not loaded correctly")
        end
        numTiresWithSound = 0
    elseif self.origIndoorVolume == 0 and self.origOutdoorVolume == 0 then
        if self.isClient then
            print("Warning: Script tireSound: sample.tires (<tires> in the tire sound-XML): 0.0 (zero) for both volumes parameters")
        end
        numTiresWithSound = 0
    elseif numTiresWithSound > 0 then
        self.multiplIndoorVolume = self.origIndoorVolume / numTiresWithSound
        self.multiplOutdoorVolume = self.origOutdoorVolume / numTiresWithSound
        
        gU_targetSelf = self
        local configId = Utils.getNoNil(self.configurations["tireSound"], 1)
        local multiplierAdjust = 0
        if configId == 2 then
            multiplierAdjust = 0.4
        elseif configId == 3 then
            multiplierAdjust = 0.5
        elseif configId == 4 then
            multiplierAdjust = 0.65
        elseif configId == 5 then
            multiplierAdjust = 0.8
        elseif configId == 6 then
            multiplierAdjust = 1.0
        elseif configId == 7 then
            multiplierAdjust = 1.25
        elseif configId == 8 then
            multiplierAdjust = 1.5
        elseif configId == 9 then
            multiplierAdjust = 1.75
        elseif configId == 10 then
            multiplierAdjust = 2.0
        elseif configId == 11 then
            multiplierAdjust = 2.5
        elseif configId == 12 then
            multiplierAdjust = 3.0
        end
        self.multiplIndoorVolume = self.multiplIndoorVolume * multiplierAdjust
        self.multiplOutdoorVolume = self.multiplOutdoorVolume * multiplierAdjust
        self.multiplVolume = math.max(self.multiplIndoorVolume, self.multiplOutdoorVolume)

        if isNokianTires == true then
            -- print("isNokianTires =================================")	
            self.multiplIndoorVolume = self.multiplIndoorVolume * 0.5
            self.multiplOutdoorVolume = self.multiplOutdoorVolume * 0.5

            local spec = self.spec_motorized
            local maxSpeed = spec.motor.maxForwardSpeed * 3.6
            if maxSpeed > 65 then
                spec.samples.tires.indoorAttributes.pitch = spec.samples.tires.indoorAttributes.pitch * 1.10
                spec.samples.tires.outdoorAttributes.pitch = spec.samples.tires.outdoorAttributes.pitch * 1.10
            elseif maxSpeed < 45 then
                spec.samples.tires.indoorAttributes.pitch = spec.samples.tires.indoorAttributes.pitch * 70 / maxSpeed
                spec.samples.tires.outdoorAttributes.pitch = spec.samples.tires.outdoorAttributes.pitch * 70 / maxSpeed
            else
                spec.samples.tires.indoorAttributes.pitch = spec.samples.tires.indoorAttributes.pitch * 75 / maxSpeed
                spec.samples.tires.outdoorAttributes.pitch = spec.samples.tires.outdoorAttributes.pitch * 75 / maxSpeed
            end
        elseif isCrawler == true then
            -- print("isCrawler =======================================")	
            self.multiplIndoorVolume = self.multiplIndoorVolume * 0.8
            self.multiplOutdoorVolume = self.multiplOutdoorVolume * 0.8
            local spec = self.spec_motorized
            spec.samples.tires.indoorAttributes.pitch = spec.samples.tires.indoorAttributes.pitch * 1.2
            spec.samples.tires.outdoorAttributes.pitch = spec.samples.tires.outdoorAttributes.pitch * 1.2
        elseif isCareWheel == true then
            -- print("isCrareWheel =====================================")	
            self.multiplIndoorVolume = self.multiplIndoorVolume * 0.65
            self.multiplOutdoorVolume = self.multiplOutdoorVolume * 0.65
            local spec = self.spec_motorized
            spec.samples.tires.indoorAttributes.pitch = spec.samples.tires.indoorAttributes.pitch * 1.13
            spec.samples.tires.outdoorAttributes.pitch = spec.samples.tires.outdoorAttributes.pitch * 1.13
        end

    end    
    self.numTiresWithSound = numTiresWithSound

    gU_lastLoadTime = g_currentMission.time
    gU_loadFilename = self.xmlFile.filename
    gU_loadHandle = self.xmlFile.handle
end

function tireSound:onStartMotor()
    local spec = self.spec_motorized
    if self.isClient then
        if self.numTiresWithSound > 1 then
            if self.multiplVolume > 0 then
                g_soundManager:playSample(spec.samples.tires, 0)
            end
        end
    end
end

function tireSound:onStopMotor()
    local spec = self.spec_motorized
    if self.isClient then
        if self.numTiresWithSound > 1 then
            if self.multiplVolume > 0 then
                g_soundManager:stopSample(spec.samples.tires)
            end
        end
    end
end

function tireSound:onUpdate(dt)
    local spec = self.spec_wheels
    local mudTiresOnStreet = 0
    local targetIndoorVolume
    local targetOutdoorVolume

    if self.numTiresWithSound > 1 and self.isClient then
        if self.multiplVolume > 0 and self:getLastSpeed() > 23 then
            
            -- How much mud tires are on the street?
            for _,wheel in pairs(spec.wheels) do
                if wheel.tireTrackAtlasIndex == 0 or wheel.tireTrackAtlasIndex == 8 or wheel.tireType == 4 or wheel.tireType == 1 then
                    local isOnField = wheel.densityType ~= 0
                    local isOnAsphalt = wheel.contact == Wheels.WHEEL_OBJ_CONTACT
                    -- renderText(0.2, 0.40, 0.05,tostring(isOnAsphalt))
                    -- renderText(0.2, 0.45, 0.05,tostring(wheel.lastTerrainAttribute))
                    if isOnAsphalt then
                        mudTiresOnStreet = mudTiresOnStreet + 1
                    elseif not isOnField then
                        local groundType = WheelsUtil.getGroundType(isOnField, wheel.contact ~= Wheels.WHEEL_GROUND_CONTACT, wheel.lastColor[4])
                        --renderText(0.2, 0.20, 0.05,tostring(groundType))
                        if groundType == 1 then
                            if wheel.lastTerrainAttribute == 7 then  --  7 == Concrete slabs / Betonplatten
                                mudTiresOnStreet = mudTiresOnStreet + 0.63
                            end
                        elseif groundType == 2 then  --  2 == HARD_TERRAIN
                            if wheel.lastTerrainAttribute == 1 then
                                mudTiresOnStreet = mudTiresOnStreet + 0.63
                            end
                        end
                    end
                end
            end
            
            -- Set the sound volume, dependig how much mud tires are on the street
            targetIndoorVolume = mudTiresOnStreet * self.multiplIndoorVolume
            targetOutdoorVolume = mudTiresOnStreet * self.multiplOutdoorVolume
            local sample = self.spec_motorized.samples
            if targetOutdoorVolume ~= sample.tires.outdoorAttributes.volume or targetIndoorVolume ~= sample.tires.indoorAttributes.volume then
                local smoothVolume
                if targetOutdoorVolume * 1.0001 < sample.tires.outdoorAttributes.volume then
                    sample.tires.indoorAttributes.volume = targetIndoorVolume
                    sample.tires.outdoorAttributes.volume = targetOutdoorVolume
                elseif mudTiresOnStreet < self.numTiresWithSound * 0.6 then
                    sample.tires.indoorAttributes.volume = targetIndoorVolume
                    sample.tires.outdoorAttributes.volume = targetOutdoorVolume
                else
                    local smooth = 1 + dt / 1400   -- smooth = approx. 1.012 at 60 FPS
                    smoothVolume = math.min(sample.tires.indoorAttributes.volume * smooth, targetIndoorVolume)
                    sample.tires.indoorAttributes.volume = math.max(smoothVolume, self.multiplIndoorVolume)
                    smoothVolume = math.min(sample.tires.outdoorAttributes.volume * smooth, targetOutdoorVolume)
                    sample.tires.outdoorAttributes.volume = math.max(smoothVolume, self.multiplOutdoorVolume)
                end
            end
        end
    end
end

function tireSound:saveToXMLFile(xmlFile, key, usedModNames)
    if self.configurations ~= nil then
        if self.configurations["tireSound"] ~= nil then
            xmlFile:setValue(key.."#configurationId", self.configurations["tireSound"])
        end
    end
end

function tireSound:onPreDelete()
    local deleteTimeGap = g_currentMission.time - gU_lastDeleteTime
    local loadTimeGap = g_currentMission.time - gU_lastLoadTime

    if deleteTimeGap < 250 then
        if loadTimeGap < 250 then
            if gU_targetSelf ~= nil then
                if gU_deleteFilename == self.xmlFile.filename then
                    if gU_loadFilename == self.xmlFile.filename then
                        if gU_deleteHandle ~= self.xmlFile.handle then
                            if gU_loadHandle ~= self.xmlFile.handle then
                                if gU_loadHandle ~= gU_deleteHandle then
                                    gU_targetSelf.configurations["tireSound"] = self.configurations["tireSound"]
                                    gU_targetSelf.speedLimit = self.speedLimit
                                end
                            end
                        end
                    end
                end
            end
        end
    end

    gU_lastDeleteTime = g_currentMission.time
    gU_deleteFilename = self.xmlFile.filename
    gU_deleteHandle = self.xmlFile.handle
end
