﻿-- Author: TeeKoo/Truck-King
-- Name: PlayerManager.lua
-- Description: Class for handling players positions

-- Modifications only with my permission, Perkele!

-- Changelog:
--  v1.0.0.0 (2.12.2021)
--		- FS22 Converting
--		- Vehicle controller is now detected with vehicle:getOwner()
--
--  v0.0.8.0 (23.1.2021)
--		- debug mode
--
-- 	v0.0.0.2 (20.1.2021):
--		- nil check to update
--		- translated log texts
-- 	v0.0.0.1 (20.1.2021):
--		- base script
--
-- ToDo:
--
PlayerManager = {};
PlayerManager_mt = Class(PlayerManager)

function PlayerManager:new(debugMode)
    local self = setmetatable({}, PlayerManager_mt)
	self.players = {};
	self.debugMode = debugMode;
    return self
end

function PlayerManager:delete()
	self.players = nil;
end

function PlayerManager:getPlayerPosition(userId)
	local uniqueUserId = self.getUniqueUserId(userId)
	if uniqueUserId ~= "" and uniqueUserId ~= nil then
		if self.players[uniqueUserId] ~= nil then
			return self.players[uniqueUserId].x, self.players[uniqueUserId].y, self.players[uniqueUserId].z
		end
	end
end

function PlayerManager.getUniqueUserId(userId)
	local user = g_currentMission.userManager:getUserByUserId(userId)
	if user ~= nil then
		return user.uniqueUserId
	end
end

function PlayerManager:updatePlayerPosition(userId, x, y, z, userName)
	local uniqueUserId = self.getUniqueUserId(userId)
	self:updatePlayerPositionWithUq(uniqueUserId, x, y, z, userName)
end

function PlayerManager:updatePlayerPositionWithUq(uniqueUserId, x, y, z, userName)
	if uniqueUserId ~= "" and uniqueUserId ~= nil then
		if self.players[uniqueUserId] == nil then
			self.players[uniqueUserId] = {}
		end
		
		if userName ~= nil then
			self.players[uniqueUserId].userName = userName		
		end
		if x ~=nil and y~= nil and z~= nil then
			self.players[uniqueUserId].x = x
			self.players[uniqueUserId].y = y
			self.players[uniqueUserId].z = z
		else
			--print("ERROR: Trying set player position to nil")			
		end
	end
end

function PlayerManager:updatePlayerPositions()
	local players = {}
	for _, player in pairs(g_currentMission.players) do -- check all players
		local name = g_currentMission.userManager:getUserByUserId(player.userId):getNickname()
		local x, y, z = player:getPositionData()
		if y > -190 then -- player y = -200 when inside vehicle
			--self.printDebug(string.format("Player: %s is standing in position X:%d Y:%d Z:%d", name, x, y, z), self.debugMode)
			self:updatePlayerPosition(player.userId,x, y, z, name)		
		else
			players[player.userId] = name -- save player id to table, so its easier to found in the next step
		end
	end	

	for index,vehicle in pairs (g_currentMission.controlledVehicles) do -- check controlled vehicles to calculate players exitpoints.
		local userId = nil

		if vehicle.getOwner ~= nil then
            local user = g_currentMission.userManager:getUserByConnection(vehicle:getOwner())
            if user ~= nil then
                userId = user:getId()
            end
		end

		if userId ~= nil then
			if players[userId] ~= nil then
				local x, y, z = self.calculateExitPoint(vehicle)
				
				if x ~= nil then
					--self.printDebug(string.format("Player %s is in a vehicle '%s'. Position (X:%d, Y:%d, Z:%d)", players[userId], vehicle:getName(), x, y, z), self.debugMode)
					self:updatePlayerPosition(userId,x, y, z, players[userId])
					players[userId] = nil
				end
			else
				--self.printDebug(string.format("Player %s is in a vehicle '%s', but has also valid standing position", g_currentMission.userManager:getUserByUserId(userId):getNickname(), vehicle:getName()), self.debugMode)
			end
		else
			--self.printDebug(string.format("Vehicle %s has incorrect parameters",vehicle:getName()), self.debugMode)
		end
	end
	
	-- if self.debugMode then
		-- for _, name in pairs(players) do -- check if players left
			-- self.printDebug(string.format("Position for player '%s' not found",name), self.debugMode)
		-- end
	-- end
end

function PlayerManager.calculateExitPoint(exitVehicle)
    local exitPoint
    if exitVehicle.getExitNode ~= nil then
        exitPoint = exitVehicle:getExitNode()
    end
    local x, y, z = getWorldTranslation(exitPoint)
    local terrainHeight = getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode, x, 300, z)
    y = math.max(terrainHeight + 0.1, y + 0.9)

	return x, y, z
end

function PlayerManager:savePlayerPositions(xmlFile)
	self:updatePlayerPositions()
	local count = 0
	for uniqueUserId,data in pairs (self.players) do
		if data ~= nil then
			if data.x ~= nil and data.y ~= nil and data.z ~= nil then
				setXMLString(xmlFile, string.format("playerPositions.player(%d)#playerName", count), Utils.getNoNil(data.userName, "UNKNOWN"))
				setXMLString(xmlFile, string.format("playerPositions.player(%d)#uniqueUserId", count), uniqueUserId)
				setXMLFloat(xmlFile, string.format("playerPositions.player(%d)#x", count), data.x)
				setXMLFloat(xmlFile, string.format("playerPositions.player(%d)#y", count), data.y)
				setXMLFloat(xmlFile, string.format("playerPositions.player(%d)#z", count), data.z)
				count = count + 1
			end
		end
	end
end

function PlayerManager.printDebug(text, debugMode)
	if debugMode then
		print(text)
	end
end